package com.titan.travelagent;

import static javax.ejb.TransactionAttributeType.NOT_SUPPORTED;
import static javax.ejb.TransactionAttributeType.REQUIRED;
import static javax.persistence.PersistenceContextType.EXTENDED;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.EJB;
import javax.ejb.EJBException;
import javax.ejb.Remove;
import javax.ejb.Stateful;
import javax.ejb.TransactionAttribute;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import com.titan.domain.Address;
import com.titan.domain.Cabin;
import com.titan.domain.Cruise;
import com.titan.domain.CruiseCabin;
import com.titan.domain.Customer;
import com.titan.domain.Reservation;
import com.titan.processpayment.CreditCardDO;
import com.titan.processpayment.ProcessPaymentLocal;

@Stateful
@TransactionAttribute(NOT_SUPPORTED)
public class TravelAgentBean implements TravelAgentRemote 
{
   @PersistenceContext(unitName="titan", type=EXTENDED)
   private EntityManager entityManager;

   @EJB private ProcessPaymentLocal processPayment;

   private Customer customer;
   private Cruise cruise;
   private Cabin cabin;
   
   public Customer findOrCreateCustomer(String first, String last) {
      try {
         Query q = entityManager.createQuery("SELECT c from Customer c where c.firstName = :first and c.lastName = :last");
         q.setParameter("first", first);
         q.setParameter("last", last);
         this.customer = (Customer)q.getSingleResult();
      } catch (NoResultException notFound) {
         this.customer = new Customer();
         this.customer.setFirstName(first);
         this.customer.setLastName(last);
         entityManager.persist(this.customer);
      }
      return this.customer;
   }

   public void updateAddress(Address addr) {
      this.customer.setAddress(addr);
      this.customer = entityManager.merge(customer);
   }

   public void setCabinID(int cabinID) 
   {
      this.cabin = entityManager.find(Cabin.class, cabinID);
      if (cabin == null) throw new NoResultException("Cabin not found");
   }

   public void setCruiseID(int cruiseID) 
   {
      this.cruise = entityManager.find(Cruise.class, cruiseID);
      if (cruise == null) throw new NoResultException("Cruise not found");
   } 

   public TicketDO bookPassage(CreditCardDO card, double price)
      throws IncompleteConversationalState 
   {
                   
      if (customer == null || cruise == null || cabin == null) 
      {
         throw new IncompleteConversationalState( );
      }

      Reservation reservation = new Reservation(customer, cruise, cabin, price, new Date( ));
      entityManager.persist(reservation);
                
      Query getCruiseCabin = entityManager.createQuery("SELECT cc FROM CruiseCabin cc WHERE " +
                                                       "cc.cabin = :cabin AND cc.cruise = :cruise");
      getCruiseCabin.setParameter("cabin", cabin);
      getCruiseCabin.setParameter("cruise", cruise);
      CruiseCabin cc = (CruiseCabin)getCruiseCabin.getSingleResult();
        
      if (cc.getIsReserved())
         throw new EJBException ("Cabin is already reserved");
      cc.setIsReserved(true);

      try 
      {
         processPayment.byCredit(customer, card, price);
     
         TicketDO ticket = new TicketDO(customer, cruise, cabin, price);
         return ticket;
      } 
      catch(Exception e) 
      {
         throw new EJBException(e);
      }
   }

   @Remove
   @TransactionAttribute(REQUIRED)
   public void checkout()
   {
      // this really isn't necessary as the entityManager would
      // be enlisted with the transaction anyways.
      entityManager.flush();
   }

}
